/**
 * \file: level_configuration.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include <unistd.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include "model/level_configuration.h"
#include "model/signature_db.h"
#include "control/configuration.h"
#include "util/logger.h"
#include "util/helper.h"

//-------------------------------------- private structures ----------------------------------------------------------
typedef struct level_configuration_t
{
	security_level_t level_number;
	bool is_permanent_level;
	feature_t *first_feature;
} level_configuration_t;

typedef struct feature_t
{
	bool active;
	feature_t *next_feature;
	char feature_name[];
} feature_t;

typedef struct level_list_item_t
{
	security_level_t security_level;
	bool has_public_key;
	level_list_item_t *next_item;
} level_list_item_t;
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- private attributes ----------------------------------------------------------
static level_list_item_t *level_list_first_item=NULL;
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- private function declaration ------------------------------------------------
static error_code_t level_configuration_check_and_set_level_dir(char *dir, size_t dir_size_max, const char *root_dir,
		security_level_t level_number);
static error_code_t level_configuration_load_configuration_file(const char *level_dir_abs, level_configuration_t *level_conf);
static feature_t *level_configuration_process_level_conf_line(char *line_in_file,int line_no);
static error_code_t level_configuration_parse_level_type(bool *is_persistent_ptr, char **line_ptr,
		size_t *line_size_ptr ,FILE *file, int *line_no_ptr);

static error_code_t level_configuration_insert_level_entry(security_level_t level,bool has_pub_key);

static feature_t *level_configuration_feature_new(const char *name, bool active);
static void level_configuration_feature_free(feature_t *feature);

//--------------------------------------------------------------------------------------------------------------------

//------------------------------------------- init / deinit -----------------------------------------------------
error_code_t level_configuration_init(const char *root_dir)
{
	DIR *dp;
	struct dirent *ep;
	char level_scripts_path[PATH_MAX];
	error_code_t result=RESULT_OK;

	snprintf(level_scripts_path,PATH_MAX,"%s%s/",root_dir,LEVEL_SUBDIR);

	dp = opendir (level_scripts_path);
	if (dp == NULL)
	{
		logger_log_error("Unable to open the level configuration in \'%s\'- %s.",root_dir,strerror(errno));
		return RESULT_INVALID;
	}

	while ((ep = readdir (dp))!=NULL && result==RESULT_OK)
	{
		security_level_t level;
		char *parse_result;
		bool has_pub_key;

		if (strcmp(ep->d_name,"..")==0 || strcmp(ep->d_name,".")==0) continue;

		if (ep->d_type != DT_DIR)
		{
			logger_log_info("Unexpected element found in feature conf directory: \'%s\'. Ignoring it.",ep->d_name);
			continue;
		}

		level=(security_level_t)strtol(ep->d_name,&parse_result,10);
		if (ep->d_name == parse_result || parse_result[0]!='\0')
		{
			logger_log_info("WARNING: Invalid level sub directory: %s. The name of a level sub directory"""
					" needs to be a positive number only. Ignoring the sub directory.",ep->d_name);
			continue;
		}

		logger_log_debug("LEVEL_CONFIGURATION - Security Level \'%d\' found.",level);
		has_pub_key=level_configuration_check_pubkey(level_scripts_path,level);
		if (has_pub_key)
			logger_log_debug("Level configuration contains a public key.");
		else
			logger_log_debug("Level configuration does not contain a public key.");

		result=level_configuration_insert_level_entry(level, has_pub_key);
	}

	(void) closedir (dp);

	return result;
}

void level_configuration_deinit(void)
{
	level_list_item_t *tmp=level_list_first_item;
	while(tmp!=NULL)
	{
		level_list_first_item=tmp->next_item;
		free(tmp);
		tmp=level_list_first_item;
	}
}

static error_code_t level_configuration_insert_level_entry(security_level_t level,bool has_pub_key)
{
	level_list_item_t *new_item;
	level_list_item_t *tmp_item;
	new_item=malloc(sizeof(level_list_item_t));
	if (new_item==NULL)
		return RESULT_NORESOURCE;

	new_item->security_level=level;
	new_item->has_public_key=has_pub_key;
	new_item->next_item=NULL;

	//no one in list, we put the new in and are done
	if (level_list_first_item==NULL)
	{
		level_list_first_item=new_item;
		return RESULT_OK;
	}

	//first one is already larger then the new one, we are putting it at the beginning of the list and are done
	if (level_list_first_item->security_level > level)
	{
		new_item->next_item=level_list_first_item;
		level_list_first_item=new_item;
		return RESULT_OK;
	}

	//not the first in list and the first one is larger then the new one -> try to find the one where the new fits in
	tmp_item=level_list_first_item;
	while (tmp_item->next_item!=NULL)
	{
		//next one of the current one is larger -> so we found the element after which we have to insert the new one
		if (tmp_item->next_item->security_level > level)
		{
			new_item->next_item=tmp_item->next_item;
			tmp_item->next_item=new_item;
			return RESULT_OK;
		}
		tmp_item=tmp_item->next_item;
	}

	//when we got here, we reached the end of the list. So we put the new item at the end
	tmp_item->next_item=new_item;

	return RESULT_OK;
}
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- level list ------------------------------------------------------------------
level_list_item_t *level_configuration_get_list_item(security_level_t level)
{
	level_list_item_t *item=level_list_first_item;
	while (item!=NULL)
	{
		if (item->security_level==level)
			return item;
		item=item->next_item;
	}

	return NULL;
}

level_list_item_t *level_configuration_get_next_list_item(level_list_item_t *level_list_item)
{
	if (level_list_item==NULL)
		return NULL;
	return level_list_item->next_item;
}

security_level_t level_configuration_list_item_get_level(level_list_item_t *level_list_item)
{
	return level_list_item->security_level;
}

bool level_configuration_list_item_has_level_pubkey(level_list_item_t *level_list_item)
{
	return level_list_item->has_public_key;
}
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------- load and validate function definitions --------------------------------------
error_code_t level_configuration_load_level(level_configuration_t **level_conf_ptr, const char *root_dir,
		security_level_t level_number)
{
	error_code_t result;
	char level_dir_abs[PATH_MAX];
	level_configuration_t *new_level_conf;

	result=level_configuration_check_and_set_level_dir(level_dir_abs, PATH_MAX, root_dir, level_number);
	if (result!=RESULT_OK)
		return result;

	new_level_conf = malloc(sizeof(level_configuration_t));
	if (new_level_conf==NULL)
		return RESULT_NORESOURCE;
	new_level_conf->level_number=level_number;
	new_level_conf->first_feature=NULL;

	result=level_configuration_load_configuration_file(level_dir_abs, new_level_conf);

	if (result!=RESULT_OK)
		free(new_level_conf);
	else
		*level_conf_ptr=new_level_conf;
	return result;
}

void level_configuration_free(level_configuration_t *level_conf)
{
	feature_t *feature;
	if (level_conf==NULL)
		return;

	feature=level_conf->first_feature;

	while (feature != NULL)
	{
		level_conf->first_feature=feature->next_feature;
		level_configuration_feature_free(feature);
		feature=level_conf->first_feature;
	}

	free(level_conf);
}
//--------------------------------------------------------------------------------------------------------------

//------------------------------------------- level entry iterator ---------------------------------------------
const feature_t *level_configuration_get_first_feature(const level_configuration_t *level_conf)
{
	if (level_conf==NULL)
		return NULL;

	return level_conf->first_feature;
}

feature_t *level_configuration_next_feature(const feature_t *feature)
{
	if (feature!=NULL)
		return feature->next_feature;
	else
		return NULL;
}
//--------------------------------------------------------------------------------------------------------------

//------------------------------------------- attribute getters ------------------------------------------------
const char *level_configuration_feature_name(const feature_t *feature)
{
	if (feature==NULL)
		return NULL;
	else
		return feature->feature_name;
}

bool level_configuration_is_feature_active(const feature_t *feature)
{
	if (feature==NULL)
		return NULL;
	else
		return feature->active;
}

security_level_t level_configuration_get_security_level(const level_configuration_t *level_conf)
{
	if (level_conf==NULL)
		return 0;
	else
		return level_conf->level_number;
}

void level_configuration_get_key_path_of_level(char *key_path,size_t path_max,const char *root_dir,
		security_level_t level)
{
	snprintf(key_path, path_max, "%s%s/%d/%s",root_dir,LEVEL_SUBDIR,(int)level,LEVEL_KEY_NAME PUBLIC_KEY_EXT);
}

bool level_configuration_check_pubkey(const char *level_scripts_path, security_level_t level)
{
	char pubkey_file_path[PATH_MAX];
	struct stat stat_result;

	snprintf(pubkey_file_path,PATH_MAX,"%s/%d/%s",level_scripts_path,level,LEVEL_KEY_NAME PUBLIC_KEY_EXT);
	return stat(pubkey_file_path,&stat_result)==0;
}

bool level_configuration_is_permanent_level(const level_configuration_t *level_conf)
{
	if (level_conf==NULL)
		return false;
	return level_conf->is_permanent_level;
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------------- feature script path generation -----------------------------------
void level_configuration_get_prepare_script_path(char *path, size_t max_path_len,
		const level_configuration_t *level_conf, change_type_t type, const char *root_dir)
{
	if (type==ALD_NORMAL_CHANGE || type==ALD_RECOVERY_ON_TIMEOUT)
		snprintf(path,max_path_len,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,(int)level_conf->level_number,
				PREPARE_LEVEL_CHANGE_FN);
	else
		snprintf(path,max_path_len,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,(int)level_conf->level_number,
				PREPARE_LEVEL_CHANGE_RECO_FN);
}

void level_configuration_get_finalize_script_path(char *path, size_t max_path_len,
		const level_configuration_t *level_conf, change_type_t type, const char *root_dir)
{
	if (type==ALD_NORMAL_CHANGE || type==ALD_RECOVERY_ON_TIMEOUT)
		snprintf(path,max_path_len,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,(int)level_conf->level_number,
				FINALIZE_LEVEL_CHANGE_FN);
	else
		snprintf(path,max_path_len,"%s%s/%d/%s",root_dir,LEVEL_SUBDIR,(int)level_conf->level_number,
				FINALIZE_LEVEL_CHANGE_RECO_FN);
}

void level_configuration_get_volatile_feature_script_path(char *path, size_t max_path_len,
		const feature_t *feature, const char *root_dir)
{
	struct stat stat_result;

	if (feature->active)
		snprintf(path,max_path_len,"%s%s/%s/%s",root_dir,FEATURE_SUBDIR,feature->feature_name,
				FEATURE_ENABLE_VOLA_DIR);
	else
		snprintf(path,max_path_len,"%s%s/%s/%s",root_dir,FEATURE_SUBDIR,feature->feature_name,
				FEATURE_DISABLE_VOLA_DIR);

	if (stat(path,&stat_result)==-1)
	{
		strncat(path,SCRIPT_EXTN,PATH_MAX-1);
	}
}

void level_configuration_get_permanent_feature_script_path(char *path, size_t max_path_len,
		const feature_t *feature, const char *root_dir, bool is_permanent_level)
{
	struct stat stat_result;

	if (feature->active && is_permanent_level)
		snprintf(path,max_path_len,"%s%s/%s/%s",root_dir,FEATURE_SUBDIR,feature->feature_name,
			FEATURE_ENABLE_PERM_DIR);
	else
		snprintf(path,max_path_len,"%s%s/%s/%s",root_dir,FEATURE_SUBDIR,feature->feature_name,
				FEATURE_DISABLE_PERM_DIR);

	if (stat(path,&stat_result)==-1)
	{
		strncat(path,SCRIPT_EXTN,PATH_MAX-1);
	}
}
//--------------------------------------------------------------------------------------------------------------

//------------------------------------- private members --------------------------------------------------------------
static error_code_t level_configuration_check_and_set_level_dir(char *dir, size_t dir_size_max, const char *root_dir,
		security_level_t level_number)
{
	struct stat stat_buf;
	char tmp_dir[PATH_MAX];
	snprintf(tmp_dir, PATH_MAX, "%s%s/%d",root_dir, LEVEL_SUBDIR,(int)level_number);

	//check for the level sub directory
	if (stat(tmp_dir, &stat_buf)!=0)
	{
		logger_log_error("Level configuration sub directory not found (%s).",tmp_dir);
		return RESULT_INVALID_LEVEL;
	}

	if (!S_ISDIR(stat_buf.st_mode))
	{
		logger_log_error("Level configuration sub directory not found (%s).",tmp_dir);
		return RESULT_INVALID_LEVEL;
	}

	strncpy(dir,tmp_dir, dir_size_max);
	return RESULT_OK;
}

static error_code_t level_configuration_load_configuration_file(const char *level_dir_abs, level_configuration_t *level_conf)
{
	FILE *file;
	char level_path[PATH_MAX];
	char *line_in_file=NULL;
	size_t line_buf_size=0;
	int line_no=1;
	error_code_t result=RESULT_OK;

	//sorry, lin(t) forces me to do so
	level_configuration_t *lintisfaction;
	lintisfaction=level_conf;

	snprintf(level_path, PATH_MAX, "%s/%s",level_dir_abs, LEVEL_CONF_FN);

	file=fopen(level_path,"r");
	if (file==NULL)
	{
		logger_log_error("Unable to read the level configuration file. %s", strerror(errno));
		result=RESULT_INVALID_LEVEL_CONFIGURATION;
	}

	//first (none comment) line contains "[transient level]" or "[persistent level]" telling the ald
	//if we have a persistent or transient level in the configuration file
	if (result==RESULT_OK)
		result=level_configuration_parse_level_type(&level_conf->is_permanent_level, &line_in_file,
				&line_buf_size,file, &line_no);

	while (getline(&line_in_file,&line_buf_size,file)!=-1 && result==RESULT_OK)
	{
		feature_t *new_feature;
		new_feature=level_configuration_process_level_conf_line(line_in_file,line_no);
		if (new_feature !=NULL)
		{
			new_feature->next_feature=lintisfaction->first_feature;
			lintisfaction->first_feature=new_feature;
			if (new_feature->active)
				logger_log_debug("LEVEL_CONFIGURATION - feature \'%s\' is active.", new_feature->feature_name);
			else
				logger_log_debug("LEVEL_CONFIGURATION - feature \'%s\' is inactive.", new_feature->feature_name);
		}
		line_no++;
	}

	if (line_in_file!=NULL)
		free(line_in_file);

	if (file!=NULL)
		fclose(file);

	return result;
}

static error_code_t level_configuration_parse_level_type(bool *is_persistent_ptr, char **line_ptr,
		size_t *line_size_ptr ,FILE *file, int *line_no_ptr)
{
	error_code_t result=RESULT_OK;
	bool is_empty;
	bool has_found=false;
	char *clean_line;

	do
	{
		if (getline(line_ptr,line_size_ptr,file)==-1)
		{
			logger_log_error("Configuration file invalid. Expecting level type tag as first none comment line.");
			result=RESULT_INVALID_LEVEL_CONFIGURATION;
		}
		else
		{
			//modifies the string in line_in_conf and returns a pointer to somewhere in the line
			//the result ensures that we have no leading spaces, no trailing spaces and no newline in the string
			clean_line=helper_trim(*line_ptr, &is_empty);

			//do we have a comment here or is the line empty?
			if (clean_line[0]!='#' && !is_empty)
			{
				has_found=true;
				if (strcmp(clean_line, TRANSIENT_LEVEL_TAG)==0)
				{
					*is_persistent_ptr=false;
				}
				else if (strcmp(clean_line, PERMANENT_LEVEL_TAG)==0)
				{
					*is_persistent_ptr=true;
				}
				else
				{
					logger_log_error("Line %d invalid. Expecting level type tag as first none comment line.",*line_no_ptr);
					result=RESULT_INVALID_LEVEL_CONFIGURATION;
				}
				(*line_no_ptr)++;
			}
		}
	}
	while(result==RESULT_OK && !has_found);

	return result;
}

static feature_t *level_configuration_process_level_conf_line(char *line_in_file,int line_no)
{
	char *feature_name;
	char *state_str;
	bool active;

	char *clean_line;
	bool is_empty=true;

	//modifies the string in line_in_conf and returns a pointer to somewhere in the line
	//the result ensures that we have no leading spaces, no trailing spaces and no newline in the string
	clean_line=helper_trim(line_in_file, &is_empty);

	//do we have a comment here or is the line empty?
	if (clean_line[0]=='#' || is_empty)
		return NULL;

	//try to find ':'
	state_str=strchr(clean_line,':');
	if (state_str==NULL)
	{
		logger_log_error("Line %d invalid. Ignoring it. Expected format: <feature name> : active | inactive",line_no);
		return NULL;
	}

	//create two strings
	state_str[0]='\0';
	state_str++;
	state_str=helper_trim(state_str,&is_empty);
	if (is_empty)
	{
		logger_log_error("Line %d invalid. Ignoring it. Expected format: <feature name> : active | inactive",line_no);
		return NULL;
	}

	feature_name=helper_trim(clean_line,&is_empty);
	if (is_empty)
	{
		logger_log_error("Line %d invalid. Ignoring it. Expected format: <feature name> : active | inactive",line_no);
		return NULL;
	}

	if (strcasecmp(state_str, "inactive")==0)
		active=false;
	else if (strcasecmp(state_str, "active")==0)
		active=true;
	else
	{
		logger_log_error("Line %d invalid. Ignoring it. Expected format: <feature name> : active | inactive",line_no);
		return NULL;
	}

	return level_configuration_feature_new(feature_name,active);
}

static feature_t *level_configuration_feature_new(const char *name, bool active)
{
	feature_t *feature;
	size_t mem_needed;

	mem_needed=sizeof(feature_t)+strlen(name)+1;
	feature=malloc(mem_needed);
	if (feature!=NULL)
	{
		feature->active=active;
		feature->next_feature=NULL;
		strcpy(feature->feature_name,name);
	}

	return feature;
}

static void level_configuration_feature_free(feature_t *feature)
{
	if (feature==NULL)
		free(feature);
}
//--------------------------------------------------------------------------------------------------------------------
